# -*- coding: utf-8 -*-

#Librerías
import re
import unicodedata
from pathlib import Path
import logging

import pdfplumber
import pandas as pd
import numpy as np
from utils import normalizar_texto, y_corte, es_tabla_excepcion, limpiar_valor, buscar_encabezado_simple, unir_textos

#Configuración del log
logging.raiseExceptions = False
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(levelname)s - %(message)s",
    handlers=[
        logging.FileHandler("extraccion_medidas_EIA.log", mode="w", encoding="utf-8"),
        logging.StreamHandler(),
    ],
)

# Definición de frases triggers para determinar las secciones del documento desde donde se extraen las tablas
TRIGGERS = [
    "que del proceso de evaluacion de impacto ambiental del proyecto puede concluirse que las siguientes medidas",
    "que del proceso de evaluacion ambiental del proyecto puede concluirse que las siguientes medidas",
]
TRIGGER_PATTERNS = [normalizar_texto(t).replace(" ", "") for t in TRIGGERS]

END_PHRASE = ["que el plan de seguimiento de las variables ambientales relevantes"]

END_PATTERNS = [normalizar_texto(t).replace(" ", "") for t in END_PHRASE]


# Función para identificar las páginas donde estan los triggers
def find_pages(pdf):
    """Devuelve (start, end) de las páginas de interés."""
    start = end = None
    for i, page in enumerate(pdf.pages, 1):
        norm = normalizar_texto(page.extract_text() or "").replace(" ", "")
        if start is None and any(t in norm for t in TRIGGER_PATTERNS):
            start = i
        if start and any(t in norm for t in END_PATTERNS):
            end = i
            break
    if start is None:
        return None, None
    return start, end or len(pdf.pages)

# Diccionario de campos a leer en cada tabla como claves canónicas y sus variantes
CANON = {
    "tipo_de_medida": [
        r"^tipo(?: de)? medida$",
    ],
     "fase": [
        r"^fase*"
    ],
    "componente_ambiental": [
        r"^componente*",
    ],
    "impacto_asociado": [
        r"^impacto*",
    ],
    "objetivo":[
        r"^objetivo$"
    ],
    "descripcion":[
        r"^descripcion$",r"^descripcion(?: y)? de la medida$"
    ],
    "justificacion":[
        r"^justificacion$", r"^justificacion de la medida$"
    ],

    "objetivo_descripcion_justificacion": [
        r"^objetivo *descripcion(?: y)? *justificacion$",
        r"^objetivo descripcion y$"
    ],
    "objetivo_descripcion": [
        r"^objetivo(?: y)? descripcion$",
    ],

    "descripcion_justificacion": [
        r"^descripcion(?: y)? justificacion$",
        r"^descripcion y justificacion de la medida$"
    ],
    "lugar_implementacion": [
        r"^lugar(?: de)? implementacion$",
        r"^lugar de$",
        r"^lugar*",
    ],
    "indicador": [
        r"^indicador*",
    ]
}
# Compilación de combinaciones
CANON = {k: [re.compile(p, re.I) for p in v] for k, v in CANON.items()}

#Funcion para convertir los valores encontrados a las claves del canon
def canonizar(clave_norm: str) -> str:
    """Devuelve la clave canónica si encaja en CANON; si no, la misma."""
    for canon, patrones in CANON.items():
        if any(p.match(clave_norm) for p in patrones):
            return canon
    return clave_norm

# Configuración de extracción de tablas
TABLE_CFG = {
    "vertical_strategy": "lines",
    "horizontal_strategy": "lines",
    "intersection_tolerance": 5,
}

# Lista con campos de claves canónicas a buscar en las tablas
KEEP = {
    "tipo_de_medida", "fase", "componente_ambiental", "impacto_asociado",
    "objetivo", "descripcion", "justificacion",
    "objetivo_descripcion", "descripcion_justificacion",
    "objetivo_descripcion_justificacion",
    "lugar_implementacion",
     "indicador"
}

# Excepcion de frases que no debe reconocer como encabezado o campos clave
excepcion_encabezado_base = ['que aplica']

# Función para detectar si una fila de una tabla es encabezado o no
def fila_es_encabezado(fila):
    """True cuando la fila es *probable* encabezado de medida.
    Criterios:
    1. Ese texto **no** es una etiqueta del canon (KEEP), de lo contrario es
       una fila‐etiqueta y la tabla es continuación.
    2. Texto en la 1ª celda y resto vacío.
    """

    if not fila or not str(fila[0]).strip():
        return False

    # 1) chequeo CANON primero
    key = canonizar(normalizar_texto(fila[0]))
    if key in KEEP:
        return False

    # 2) luego vemos si hay texto en columnas extras
    hay_texto = any(str(c).strip() for c in fila[1:])
    sin_none = all(str(c).strip() != "None" for c in fila[1:])

    if hay_texto and sin_none:
        return False

    if any(sub in key for sub in excepcion_encabezado_base):
        return False
    # 3) si ninguna de las condiciones anteriores se cumple, entonces sí es encabezado
    return True

# Función para unir tablas después de extraerlas
def fusionar_tablas(tables):
    """
    Fusiona tablas adyacentes salvo que:
      • la 1.ª fila sea encabezado (criterio A), o
      • la 1.ª celda repita una etiqueta KEEP ya vista en la tabla vigente (criterio B).
    """
    merged = []
    keep_vistos = set()

    for page, filas in tables:
        # Analizar qué etiquetas KEEP están presentes en esta tabla
        keys_present = set()
        for f in filas:
            if f and str(f[0]).strip():
                k = canonizar(normalizar_texto(str(f[0])))
                if k in KEEP:
                    keys_present.add(k)

        # Analizar la primera fila
        primera = filas[0][0] if filas and filas[0] else ""
        key_norm = canonizar(normalizar_texto(primera))
        es_keep = key_norm in KEEP

        # Criterios para determinar si es una nueva tabla
        es_encabezado = fila_es_encabezado(filas[0])
        repeticion_keep = es_keep and key_norm in keep_vistos

        # Decisión: ¿Nueva tabla o continuación?
        if not merged:
            nueva_tabla = True
        elif es_encabezado:
            nueva_tabla = True
        elif repeticion_keep:
            nueva_tabla = True
        else:
            nueva_tabla = False

        if nueva_tabla:
            # Crear nueva tabla
            filas_con_pagina = [(fila, page) for fila in filas]
            merged.append([page, filas_con_pagina])
            keep_vistos = keys_present.copy()
        else:
            # Fusionar con la tabla anterior
            filas_con_pagina = [(fila, page) for fila in filas]
            merged[-1][1].extend(filas_con_pagina)
            keep_vistos.update(keys_present)

    return merged

# Función para leer cada fila de cada tabla y recopilar los registros
def leer_filas_tabla(tabla: list[list[str]]) -> list[dict]:
    """
    Interpreta una tabla fusionada en filas:
      • Encabezado  = filas sin valor a la derecha.
      • Fila-etiqueta = texto en la 1.ª celda + valores a la derecha.
      • Fila-continuación = 1.ª celda vacía, valores a la derecha.
    Une automáticamente textos cortados entre páginas.
    """

    medidas, encabezado, campos = [], [], {}
    ultima_key: str | None = None        # última etiqueta válida
    pagina_inicio_medida = None
    etiquetas_vistas = set()  # etiquetas KEEP ya vistas en la tabla

    #Funcion para determinar cuando cierra una medida y empieza otra, es redundante con la función de merge tables por si no detecta correctamente el inicio de una nueva medida
    def cerrar():
        if encabezado:
            logging.info(f"🔒 CERRANDO MEDIDA: '{' '.join(encabezado).strip()}'")
            medidas.append(
                {
                    "nombre_medida": " ".join(encabezado).strip(),
                    "pagina_inicio": pagina_inicio_medida,
                    **{k: v for k, v in campos.items() if k in KEEP},
                }
            )

    # Se recorre cada tabla fila a fila
    for fila, pagina_actual in tabla:
        cierre_habilitado = True
        fila = [str(c).strip() if c else "" for c in fila]
        c0, resto = fila[0], fila[1:]
        valores = [c for c in resto if c]

        # 1) Fila - encabezado
        if c0 and not valores:
            key = canonizar(normalizar_texto(c0))

            if key in excepcion_encabezado_base:
                logging.info(f"   ➜ EXCEPCIÓN: no cortar por '{c0}'")
                cierre_habilitado = False
                continue

            if (key not in KEEP) and (key not in excepcion_encabezado_base):
                # --> es un encabezado de medida
                if campos:  # se cierra la medida anterior y comienza una nueva
                    cerrar()
                encabezado = [c0]
                campos = {}
                ultima_key = None
                etiquetas_vistas = set()
                pagina_inicio_medida = pagina_actual
                continue
            logging.info(f"   ➜ ACCIÓN: Procesando como posible encabezado")

        # 2) Fila-etiqueta  (hay texto en c0 y valores a la derecha)
        if c0 and (valores or (canonizar(normalizar_texto(c0)) in KEEP)):
            if pagina_inicio_medida is None:
                pagina_inicio_medida = pagina_actual
            key = canonizar(normalizar_texto(c0))
            valor = limpiar_valor(" ".join(valores)) if valores else ""

            if key in KEEP:
                if key in etiquetas_vistas:
                    logging.info(f"🔄 REPETICIÓN DETECTADA: '{key}' - Cerrando medida anterior")
                    if campos:
                        cerrar()
                    # Iniciar nueva medida sin encabezado explícito
                    encabezado = [f"Medida sin título (página {pagina_actual})"]
                    campos = {}
                    etiquetas_vistas = set()
                    pagina_inicio_medida = pagina_actual

                etiquetas_vistas.add(key)

                # Reglas para etiquetas combinadas que pueden estar divididas entre una fila y otra
                if key == "justificacion" and "descripcion_justificacion" in campos:
                    campos["descripcion_justificacion"] += " " + valor
                    ultima_key = "descripcion_justificacion"
                elif key == "justificacion" and "objetivo_descripcion_justificacion" in campos:
                    campos["objetivo_descripcion_justificacion"] += " " + valor
                    ultima_key = "objetivo_descripcion_justificacion"
                elif key == "descripcion" and "objetivo_descripcion" in campos:
                    campos["objetivo_descripcion"] += " " + valor
                    ultima_key = "objetivo_descripcion"
                else:
                    campos[key] = " ".join([campos.get(key, ""), valor]).strip()
                    ultima_key = key
            else:
                ultima_key = None
            logging.info(f"   ➜ ACCIÓN: Fila-etiqueta {ultima_key}")

            continue  # fila procesada

        # 3) Fila-continuación (sin etiqueta, pero con valores)
        if not c0 and valores:
            if ultima_key:
                # continuación legítima de la última key reconocida
                campos[ultima_key] += " " + limpiar_valor(" ".join(valores))
            # si no hay ultima_key (fila huérfana), simplemente la ignoramos
            logging.info(f"   ➜ ACCIÓN: Fila-continuación")
            continue

        # 4) Fila sin valores, separador
        texto = " ".join(fila).strip()
        if not texto:
            continue

        if campos and cierre_habilitado:     # ya dentro de una medida
            cerrar()                         # cerramos y empezamos otra

            encabezado, campos, ultima_key = [texto], {}, None
            etiquetas_vistas = set()
            pagina_inicio_medida = pagina_actual
        else:
            logging.info(f"   ➜ ACCIÓN: Fila separador/vacía")
            encabezado.append(texto)
            if pagina_inicio_medida is None:
                pagina_inicio_medida = pagina_actual

    cerrar()
    return medidas

# Función principal para procesar un documento pdf
def procesar_pdf(path: Path):
    """Devuelve (registros, start, end)."""
    doc_id = path.stem
    with pdfplumber.open(path) as pdf:
        start, end = find_pages(pdf)
        print(f"páginas trigger = {start}, end = {end}")
        if start is None:
            logging.info("      ¡trigger no encontrado!")
            trigger_not_found.append(doc_id)
            return [], None, None

        # 1) Extraer tablas página por página
        tablas_raw = []

        for pnum in range(start, end + 1):
            logging.info(f"Procesando página {pnum}")
            page = pdf.pages[pnum - 1]

            if pnum == start:
                logging.info(f"Aplicando recorte superior en página {pnum}")
                y0s = [y_corte(page, p, despues=True) for p in TRIGGER_PATTERNS]
                y0s = [y for y in y0s if y]
                if y0s:
                    page = page.crop((0, min(y0s), page.width, page.height))
            if pnum == end:
                logging.info(f"Aplicando recorte inferior en página {pnum}")
                y1s = [y_corte(page, p, despues=False) for p in END_PATTERNS]
                y1s = [y for y in y1s if y]
                if y1s:
                    page = page.crop((0, 0, page.width, min(y1s)))

            for tbl in page.extract_tables(TABLE_CFG):
                filas = [r for r in tbl if any(c for c in r)]
                if filas and not es_tabla_excepcion(filas):
                    tablas_raw.append((pnum, filas))

        # 2) Fusionar tablas que continúan entre paginas y separar tablas nuevas
        tablas_fusion = fusionar_tablas(tablas_raw)

        final_tables = []
        for pnum, filas_con_paginas in tablas_fusion:
            if filas_con_paginas:
                # Extraer solo las filas para el análisis de encabezado
                filas = [fila for fila, _ in filas_con_paginas]
                primera = filas[0]
                logging.info(f"\nTabla detectada en pág.{pnum}")
                logging.info(f"Primera fila: {primera}")

                if not fila_es_encabezado(primera):
                    # Buscar e insertar encabezado
                    pagina_primera_fila = filas_con_paginas[0][1]
                    cand = buscar_encabezado_simple(pdf.pages[pagina_primera_fila - 1], primera)
                    if cand:
                        nuevo_encabezado = ([cand] + [""] * (len(primera) - 1), pagina_primera_fila)
                        filas_con_paginas.insert(0, nuevo_encabezado)
                        logging.info(f"   ► encabezado insertado: {cand}")
            final_tables.append((pnum, filas_con_paginas))

        # 3) Parsear fila por fila y recopilar registros
        registros = []
        for pnum, filas_con_paginas in final_tables:
            logging.info(f"\nProcesando tabla en pág.{pnum}")
            for medida in leer_filas_tabla(filas_con_paginas):
                medida.update({"documento_id": doc_id, "pagina": medida.get("pagina_inicio", pnum)})
                if "pagina_inicio" in medida:
                    del medida["pagina_inicio"]
                registros.append(medida)

    return registros, start, end

# Funión para ejecución principal del código
if __name__ == "__main__":
    carpeta = Path("documentosRCA/EIA")    # carpeta con los documentos pdf a procesar
    #para procesar solo los documentos de tipo digitalizado
    id_df = pd.read_excel('listado_id_proyectos.xlsx', sheet_name='EIA', usecols=['id']) #listado con id de proyectos a procesar, puede ser un subconjunto de los documentos en carpeta.
    id_list = id_df['id'].astype(str).tolist()
    id_list = set(id_list)

    #pdfs solo si el nombre del archivo esta en la lista id
    pdfs = [f for f in sorted(carpeta.glob("*.pdf")) if f.stem in id_list]

    total = len(pdfs)

    logging.info(f"—  {total} documento(s) a analizar  —")
    medidas = []
    estados =[]
    trigger_not_found = []
    for idx, pdf in enumerate(pdfs, 1):
        logging.info(f"[{idx:>3}/{total}] procesando documento: {pdf.name}")

        try:
            registros, start, end = procesar_pdf(pdf)
            n = len(registros)
            estados.append({
                "documento_id": pdf.stem,
                "archivo": pdf.name,
                "status": "procesado",
                "n_medidas": n,
                "error": None
            })
        except Exception as e:
            logging.error(f"      ERROR: no se pudo procesar '{pdf.name}' (ID={pdf.stem}): {e}")
            estados.append({
                "documento_id": pdf.stem,
                "archivo": pdf.name,
                "status": "error",
                "n_medidas": 0,
                "error": f"{type(e).__name__}: {e}"
            })
            continue

        if registros:
            start_page = min(r["pagina"] for r in registros)
            end_page   = max(r["pagina"] for r in registros)
            logging.info(f"      ► páginas con medidas detectadas → inicio = {start_page}, fin = {end_page}")
        else:
            logging.info("      ► ¡no se procesó el pdf!")

        medidas.extend(registros)

    logging.info(f"\nExtracción terminada. Medidas totales: {len(medidas)}")

    df = pd.DataFrame(medidas)

    for col in df.columns:
        if col in df.columns:
            df[col] = (
                df[col].astype(str)
                    .str.replace(r"[\n\r]+", " ", regex=True)
                    .str.replace(r"\s+", " ", regex=True)
                    .str.strip()
                    .replace({"nan": np.nan})
            )

    needed_cols = [
        "objetivo_descripcion_justificacion",
        "objetivo_descripcion",
        "descripcion_justificacion",
        "objetivo", "descripcion", "justificacion",
    ]
    for col in needed_cols:
        if col not in df.columns:
            df[col] = np.nan

    # Máscara de filas para completar el campo objetivo_descripción_justificación fusionado
    mask = df["objetivo_descripcion_justificacion"].isna() | \
           df["objetivo_descripcion_justificacion"].eq("")
    df["objetivo_descripcion_justificacion"] = df["objetivo_descripcion_justificacion"].astype(object)
    # rellenar en el orden lógico O → D → J
    df.loc[mask, "objetivo_descripcion_justificacion"] = (
        df.loc[mask].apply(
            lambda row: unir_textos(
                row.get("objetivo"),                          # 1
                row.get("objetivo_descripcion"),              # 2
                row.get("descripcion"),                       # 3
                row.get("descripcion_justificacion"),         # 4
                row.get("justificacion")                      # 5
            ),
            axis=1
        )
    )

    #Para eliminar puntuaciones, espacios y palabra tabla de los nombres de medidas
    df['nombre_medida'] = df['nombre_medida'].str.replace(
        r'^(?:tabla\s*)?\d+(?:\.\d+)*\.?\s*',  # opcional "tabla", luego números/puntos, punto opcional y espacios
        '',
        regex=True,
        case=False
    )

    #Se carga información de proyectos para tabla final de resultados
    base_proyectos = pd.read_excel('informacion_proyectos.xlsx')
    base_proyectos['ID_EXPEDIENTE'] = base_proyectos['ID_EXPEDIENTE'].astype(str)

    df = df.merge(base_proyectos, left_on='documento_id', right_on='ID_EXPEDIENTE', how='left')
    df.drop(columns=['documento_id'], inplace=True)

    # Se cambian los nombres a las columnas
    df.rename(columns={
        "pagina": "PAGINA",
        "nombre_medida": "NOMBRE_MEDIDA",
        "tipo_de_medida": "TIPO_DE_MEDIDA",
        "fase": "FASE_DEL_PROYECTO",
        "componente_ambiental": "COMPONENTE_AMBIENTAL",
        "impacto_asociado": "IMPACTO_ASOCIADO",
        "objetivo_descripcion_justificacion": "OBJETIVO_DESCRIPCIÓN_Y_JUSTIFICACIÓN",
        "indicador": "INDICADOR",

    }, inplace=True)


    wanted = [
        "ID_EXPEDIENTE", "TIPO_PRESENTACION", "NOMBRE_PROYECTO", "REGION", "COMUNA", "TIPLOGIA_PROYECTO", "N°RCA", "AÑO_RCA", "ETAPA", "TIPO_ETAPA_1", "PAGINA","NOMBRE_MEDIDA", "TIPO_DE_MEDIDA", "FASE_DEL_PROYECTO", "COMPONENTE_AMBIENTAL", "IMPACTO_ASOCIADO", "OBJETIVO_DESCRIPCIÓN_Y_JUSTIFICACIÓN"
    ]

    #creamos las tablas con los estados de procesamiento y los triggers no encontrados para revisar
    df_estados = pd.DataFrame(estados)
    trigger_no_encontrados = pd.DataFrame(trigger_not_found)

    #Se crea archivo xlsx de salida
    with pd.ExcelWriter("medidas_EIA.xlsx", engine="openpyxl") as writer:
        df.reindex(columns=wanted).to_excel(writer, index=False, sheet_name="medidas")
        df_estados.to_excel(writer, index=False, sheet_name="reporte_procesamiento"),
        trigger_no_encontrados.to_excel(writer, index=False, sheet_name="trigger_no_encontrados")

    logging.info(f"✔ Se guardaron {len(df)} medidas en 'medidas_EIA.xlsx'")